function [scene,parms] = sceneCreate(sceneName,varargin)
% Create a scene structure.
%
%  [scene,parms] = sceneCreate(sceneName,varargin)
%
% A scene describes the light emitted from a planar object, say a flat
% screen display.  The scene is located at some distance from the center of
% the optics, has a field of view, and a spectral radiance distribution.
% All of these properties, and several others, can be set. A variety of
% images can be created automatically.  The routines that create these
% scenes, including this one, serve as a template for creating others. 
%
% MACBETH COLOR TEST PATTERNS
%
%   The default, scene = sceneCreate, is a Macbeth color checker illuminated
%   by a D65 light source with a mean luminance of 100 cd/m2.  The scene is
%   described only a small number of spatial 64x96 (row,col).  This can be
%   changed using the patchSize argument (default - 16 pixels).  The
%   wavelength  400:10:700 samples, making it efficient to use for experiments.   
%
%   Example:  
%    scene = sceneCreate('macbeth',32);
%
%    patchSize = 8;
%    spectrum.wave = (380:4:1068)';
%    scene = sceneCreate('macbethEE_IR',patchSize,spectrum);
%
%      {'macbethd65'}  - Create a Macbeth D65 image.  Optional
%         parameter of patch size (default = 16 pixels). 
%      {'macbethd50'}         - D50 illuminant
%      {'macbethillc'}        - Illuminant C
%      {'macbethfluorescent'} - Fluorescent illuminant
%      {'macbethtungsten'}    - Tungsten illuminant
%      {'macbethEE_IR'}       - Equal energy extends out to the IR
%
%   Use the SPD Scale function to adjust illuminants.
%     scene = sceneCreate('macbethEE_IR',patchSize,spectrum);
%     wave = sceneGet(scene,'wave');
%     curIlluminant = sceneGet(scene,'illuminant');
%     newIlluminant = blackbody(wave,3000);
%     scene = sceneSPDScale(scene,curIlluminant,'divide');
%     scene = sceneSPDScale(scene,newIlluminant,'multiply');
%
%   The size of the individual patches and the wavelength sampling are both
%   parameters. They can be set using the calling procedure
%
%         patchSizePixels = 16;
%         spectrum.wave = [380:5:720]; 
%         scene = sceneCreate('macbethTungsten',patchSizePixels,spectrum);
%
% SPATIAL TEST PATTERNS:   
%
%      {'ringsRays'}            - Resolution pattern
%      {'harmonic'}             - Harmonic
%      {'sweepFrequency'}       - Increasing frequency to the right,
%               increasing contrast upward
%      {'lined65'}              - Line with D65 spectrum
%      {'lineee''}              - Line with equal energy spectrum
%      {'pointArray'}           - Point array with D65 spectrum
%      {'gridlines'}            - Grid lines with D65 spectrum
%      {'checkerboard'}         - Checkerboard with equal photon spectrum
%      {'frequencyOrientation'} - Demosaicking test pattern, equal photon spectrum
%      {'slantedEdge'}  - Used for ISO spatial resolution, equal photon spectrum
%      {'zonePlate'}   - Circular zone plot, equal photon spectrum
%      {'starPattern'} - Radial lines used to test printers and displays 
%
%  Additional parameters are available for several of the patterns.  For
%  example, the harmonic call can set the frequency, contrast, phase,
%  angle, row and col size of the harmonic.
%
%        parms.freq = 1; parms.contrast = 1; parms.ph = 0;
%        parms.ang= 0; parms.row = 64; parms.col = 64; parms.GaborFlag=0;
%        [scene,parms] = sceneCreate('harmonic',parms);   
%
%  Many of the patterns can have an arbitrary image (row,col) size.  This
%  is possible for whitenoise, impulse1dee,lined65,
%
%         imageSize = 128; scene = sceneCreate('lined65',imageSize);
%
%  Other patterns have different parameters:
%
%         sceneCreate('slantedBar',imageSize,edgeSlope);
%         scene = sceneCreate('checkerboard',pixelsPerCheck,numberOfChecks)
%         scene = sceneCreate('gridlines',imageSize,pixelsBetweenLines);
%         scene = sceneCreate('pointarray',imageSize,pixelsBetweenPoints);
%
% NOISE ANALYSIS TEST PATTERNS
%
%      {'linearIntensityRamp'}  -
%      {'uniformEqualEnergy'}   - Equal energy
%      {'uniformEqualPhoton'}   - Equal photon density
%      {'uniformd65'}           - 
%      {'whitenoise'}           - Noise pattern for testing
%
%    The uniform patterns are small by default (32,32).  If you would like
%    them at a higher density (not much point), you can use
%        sceneCreate('uniformD65',256) 
%    where 256 is the image size in pixels.
%
% SCENES FROM IMAGES
%      It is possible to create scenes  using data in image fiels.
%      The high dynamic range, multispectral image data included in
%      data\images\MultiSpectral directory are an important source of
%      data.  It is also possible to simply read a tiff or jpeg file and
%      create a scene structure.  These image-based scenes created by the
%      call sceneFromFile which, in turn, calls this function.  
%
% Copyright ImagEval Consultants, LLC, 2003.

if ieNotDefined('sceneName'), sceneName = 'default'; end

% Identify the object type
scene.type = 'scene';

switch lower(sceneName)
    case 'default'
        % The user can make a Macbeth with different patch sizes and
        % wavelength sampling, by calling with additional arguments, such
        % as scene = sceneCreate('macbethd65',16,spectrum); 
        scene = sceneDefault(scene,'d65');
    case {'macbeth','macbethd65'}
        % sceneCreate('macbethD65',24);
        scene = sceneDefault(scene,'d65',varargin);
    case {'macbethd50'}
        scene = sceneDefault(scene,'d50',varargin);
    case {'macbethc','macbethillc'}
        scene = sceneDefault(scene,'c',varargin);
    case {'macbethfluorescent','macbethfluor'}
        scene = sceneDefault(scene,'fluorescent',varargin);
    case {'macbethtungsten','macbethtung'}
        scene = sceneDefault(scene,'tungsten',varargin);
    case {'macbethee_ir','macbethequalenergyinfrared'}   
        % Equal energy illumination into the IR
        % The way to call this would be
        % patchSize = 16;
        % spectrum.wavelength = 380:4:1068;
        % scene = sceneCreate('macbethEEIR',patchSize,spectrum)
        scene = sceneDefault(scene,'ir',varargin);
        
    % Monochrome,RGB and multispectral add only a little.  Mostly created in sceneFromFile
    case {'monochrome','unispectral'}
        % sceneMonochrome is used for images with only one spectral band.
        scene = sceneMonochrome(scene);
    case {'multispectral','hyperspectral'}
        scene = sceneMultispectral(scene); 
    case 'rgb'
        if isempty(varargin), scene = sceneRGB; 
        else scene = sceneRGB(varargin{1});end

    case {'mackay','rayimage','ringsrays'}
        % Also called the Siemens star pattern
        % radF = 24; imSize = 512;
        % vcAddAndSelectObject(sceneCreate('mackay',radF,imSize)); 
        % sceneWindow();
        radFreq = 8; sz = 256; 
        if length(varargin) >= 1, radFreq = varargin{1}; end
        if length(varargin) >= 2, sz = varargin{2}; end
        if length(varargin) >= 3
            wave  = varargin{3};
            scene = sceneSet(scene,'wave',wave);
        end
        scene = sceneMackay(scene,radFreq,sz);
        
    case {'czp'}
        
        disp('In CZP. . . ');
        % Also called the Siemens star pattern
        % radF = 24; imSize = 512;
        % vcAddAndSelectObject(sceneCreate('mackay',radF,imSize)); 
        % sceneWindow();
        radFreq = 48; sz = 256; 
        if length(varargin) >= 1, 
           radFreq = varargin{1}; 
        end
        if length(varargin) >= 2, 
           sz = varargin{2}; 
        end
        if length(varargin) >= 3
            wave  = varargin{3};
            scene = sceneSet(scene,'wave',wave);
        end
        
        radFreq = 96; sz = 512; 
        scene = sceneCzp(scene,radFreq,sz);
        
        
    case {'harmonic','sinusoid'}
        if isempty(varargin), 
            [scene,parms] = sceneHarmonic(scene);
        else
            parms = varargin{1};
            [scene,parms] = sceneHarmonic(scene,parms);
        end
    case {'sweep','sweepfrequency'}
        % These are always equal photon type.  Could add a third argument
        % for spectral type.
        % sz = 512; maxF = sz/16; sceneCreate('sweepFrequency',sz,maxF);
        sz = 128; maxFreq = sz/16;
        if length(varargin) >= 1, sz = varargin{1}; end
        if length(varargin) >= 2, maxFreq = varargin{2}; end
        scene = sceneSweep(scene,sz,maxFreq);
    case {'ramp','linearintensityramp','rampequalphoton'}
        if isempty(varargin),  sz = 32;
        else                   sz = varargin{1};
        end
        scene = sceneRamp(scene,sz);
    case {'uniform','uniformee','uniformequalenergy'}   %Equal energy
        if isempty(varargin), sz = 32; 
        else                  sz = varargin{1};
        end
        scene = sceneUniform(scene,'ee',sz);       
    case {'uniformeespecify'}   % Equal energy, specify waveband
        % scene = sceneCreate('uniformEESpecify',sz,wavelength);
        sz = 32; wavelength = 400:10:700;
        if ~isempty(varargin), sz = varargin{1}; end
        if length(varargin) > 1, wavelength = varargin{2}; end
        scene = sceneSet(scene,'wave',wavelength(:));
        scene = sceneUniform(scene,'ee',sz);
    case {'uniformequalphoton','uniformephoton'}        %Equal photon density
        % sceneCreate('uniformEqualPhoton',128);
        if isempty(varargin),  sz = 32;
        else                   sz = varargin{1};
        end
        scene = sceneUniform(scene,'ephoton',sz);
    case 'uniformd65'
        % sceneCreate('uniformEqualPhoton',64);
        % We should include an option for wavelength so that we extend into
        % the IR
        if isempty(varargin),  sz = 32;
        else                   sz = varargin{1};
        end
        scene = sceneUniform(scene,'D65',sz);
    case {'lined65','impulse1dd65'}
        if isempty(varargin), sz = 64;
        else sz = varargin{1};
        end
        scene = sceneLine(scene,'D65',sz);
    case {'lineee','impulse1dee'}
        % scene = sceneCreate('lineee',128,2);
        % scene = sceneCreate('lineee',128,2,380:4:1068);
        sz = 64; offset = 0;
        if length(varargin) >= 1, sz = varargin{1};     end
        if length(varargin) >= 2, offset = varargin{2}; end
        if length(varargin) == 3
            scene = sceneSet(scene,'wave',varargin{3}); 
        end
        scene = sceneLine(scene,'equalEnergy',sz,offset);
    case {'lineequalphoton','lineep'}
        sz = 64; offset = 0;
        if length(varargin) >= 1, sz = varargin{1};     end
        if length(varargin) >= 2, offset = varargin{2}; end
        scene = sceneLine(scene,'equalPhoton',sz,offset);
    case {'whitenoise','noise'}
        scene = sceneSet(scene,'name','white noise');
        if   isempty(varargin), scene = sceneNoise(scene);
        else scene = sceneNoise(scene,varargin{1});   % Size parameter
        end
    case {'pointarray','manypoints'}
        sz = 128; spacing = 16; spectralType = 'd65';
        if length(varargin) >= 1, sz           = varargin{1}; end
        if length(varargin) >= 2, spacing      = varargin{2}; end
        if length(varargin) >= 3, spectralType = varargin{3}; end
        scene = scenePointArray(scene,sz,spacing,spectralType);

    case {'gridlines','distortiongrid'}
        sz = 128; spacing = 16; spectralType = 'd65';
        if length(varargin) >= 1, sz           = varargin{1}; end
        if length(varargin) >= 2, spacing      = varargin{2}; end
        if length(varargin) >= 3, spectralType = varargin{3}; end
        scene = sceneGridLines(scene,sz,spacing,spectralType);

    case {'checkerboard'}
        period = 16; spacing = 8; spectralType = 'ep';
        if length(varargin) >= 1, period       = varargin{1}; end
        if length(varargin) >= 2, spacing      = varargin{2}; end
        if length(varargin) >= 3, spectralType = varargin{3}; end
        scene = sceneCheckerboard(scene,period,spacing,spectralType);

    case {'demosaictarget','freqorientpattern','frequencyorientation','freqorient'}
        %   parms.angles = linspace(0,pi/2,5);
        %   parms.freqs =  [1,2,4,8,16];
        %   parms.blockSize = 64;
        %   parms.contrast = .8;
        % scene = sceneCreate('freqorient',parms);
        if isempty(varargin), scene = sceneFOTarget(scene);
        else 
            % First argument is parms structure 
            scene = sceneFOTarget(scene,varargin{1});
        end
    case {'slantedbar','iso12233','slantededge'}
        % scene = sceneCreate('slantedEdge',128,1.33);  % size, slope
        % scene = sceneCreate('slantedEdge',128,1.33,(380:4:1064));  % size, slope, wave
        if isempty(varargin), scene = sceneSlantedBar(scene);
        elseif length(varargin) >= 2
            imSize = varargin{1}; barSlope = varargin{2};
            if length(varargin) == 3
                scene = sceneSet(scene,'wave',varargin{3});
            end
            scene = sceneSlantedBar(scene,imSize,barSlope);
        else
            error('Incorrect arguments for slanted bar.')
        end
    case {'zoneplate'}
        scene = sceneZonePlate(scene,384);
    case {'starpattern','radiallines'}
        % Thin radial lines - Useful for testing oriented blur
        %
        % scene = sceneCreate('starPattern');
        % scene = sceneCreate('starPattern',384);
        imSize = 256; spectralType = 'ee'; nLines = 8;
        if length(varargin) >=1, imSize = varargin{1}; end
        if length(varargin) >=2, spectralType = varargin{2}; end
        if length(varargin) >=3, nLines = varargin{3}; end
        scene = sceneRadialLines(scene,imSize,spectralType,nLines);

    otherwise
        error('Unknown scene format.');
end
    
% Initialize scene geometry, spatial sampling
scene = sceneInitGeometry(scene);
scene = sceneInitSpatial(scene);

% All scenes start at 100 cd/m2, assuming they contain photon data
if checkfields(scene,'data','photons') && ~isempty(scene.data.photons)
    luminance = sceneCalculateLuminance(scene);
    scene = sceneSet(scene,'luminance',luminance);
    % meanL eliminated on April 5, 2010
    % scene = sceneSet(scene,'meanLuminance',meanL);
    scene = sceneAdjustLuminance(scene,100);
end

return;

%---------------------------------------------------
function scene = sceneNoise(scene,sz,contrast)
%
% Make a white noise stimulus

if ieNotDefined('sz'), sz = 128; end
if ieNotDefined('contrast'), contrast = 20; end

scene = initDefaultSpectrum(scene,'hyperspectral');
wave = sceneGet(scene,'wave');
nWave = sceneGet(scene,'nwave');

% This is an image with reasonable dynamic range (10x).
d   = ieScale(randn(sz),1/contrast,1.0);   
d65 = vcReadSpectra('D65',wave);

data = zeros(sz(1),sz(2),length(wave));
for ii=1:nWave, data(:,:,ii) = d*d65(ii); end

% Allocate space for the (compressed) photons
scene = sceneSet(scene,'cphotons',data);

% By setting the fov here, we will not override the value in
% sceneInitSpatial() when this returns
scene = sceneSet(scene,'fov',1);

return;
        

%----------------------------------
function scene = sceneDefault(scene,illuminantType,args)
%
% Default scene is a Macbeth chart with D65 illuminant and patchSize 16
% pixels.   

if ieNotDefined('illuminantType'), illuminantType = 'd65'; end
if ieNotDefined('args'), args = []; end

if (isempty(args) || isempty(args{1})), patchSize = 16; 
else patchSize = args{1};
end

% Create the scene variable
scene = sceneSet(scene,'type','scene');
if isempty(args) || length(args) < 2 || isempty(args{2})
        scene = initDefaultSpectrum(scene,'hyperspectral');
else    scene = sceneSet(scene,'spectrum',args{2}); 
end
spectrum = sceneGet(scene,'spectrum');

switch lower(illuminantType)
    case 'd65'
        scene = sceneSet(scene,'name','Macbeth (D65)');
        lightSource = illuminantCreate('D65',[],100,spectrum);
    case 'd50'
        scene = sceneSet(scene,'name','Macbeth (D50)');
        lightSource = illuminantCreate('D50',[],100,spectrum);
    case 'fluorescent'
        scene = sceneSet(scene,'name','Macbeth (Fluorescent)');
        lightSource = illuminantCreate('Fluorescent',[],100,spectrum);
    case 'c'
        scene = sceneSet(scene,'name','Macbeth (Ill C)');
        lightSource = illuminantCreate('illuminantC',[],100,spectrum);
    case 'tungsten'
        scene = sceneSet(scene,'name','Macbeth (Tungsten)');
        lightSource = illuminantCreate('tungsten',[],100,spectrum);
    case 'ir'
        scene = sceneSet(scene,'name','Macbeth (IR)');
        lightSource = illuminantCreate('equalEnergy',[],100,spectrum);
    otherwise
        error('Unknown illuminant type.');
end

% Default distance in meters.
scene = sceneSet(scene,'distance',1.2);

% Scene magnification is always 1.  
% Optical images have other magnifications that depend on the optics.
scene = sceneSet(scene,'magnification',1.0);

% The default patch size is 16x16.  
spectrum = sceneGet(scene,'spectrum');
macbethChartObject = macbethChartCreate(patchSize,(1:24),spectrum);
scene = sceneCreateMacbeth(macbethChartObject,lightSource,scene);

% Store the scene illuminant in units of energy
illWave = sceneGet(lightSource,'wave');
energy = Quanta2Energy(sceneGet(lightSource,'wave'),sceneGet(lightSource,'photons'));
scene = sceneSet(scene,'illuminantwave',illWave);
scene = sceneSet(scene,'illuminantdata',energy(:));

return;

%--------------------------------------------------
function scene = sceneMonochrome(scene)
%
scene = sceneSet(scene,'name','monochrome');

% Probably should be eliminated.

% We set a default spectrum, but this is usually over-ridden, see
% sceneFromFile.
scene = initDefaultSpectrum(scene,'monochrome');

return;

%--------------------------------------------------
function scene = sceneMultispectral(scene)

scene = sceneSet(scene,'name','multispectral');
scene = initDefaultSpectrum(scene,'multispectral');

return;

%--------------------------------------------------
function scene = sceneRGB(scene)
%

if ieNotDefined('scene'), scene.type = 'scene'; end

scene = sceneSet(scene,'name','rgb');
scene = sceneSet(scene,'type','scene');
scene = initDefaultSpectrum(scene,'hyperspectral');

return;

%--------------------------------------------------
% function scene = sceneHyperspectral(scene)
% 
% scene = sceneSet(scene,'name','hyperspectral');
% scene = sceneSet(scene,'type','scene');
% scene = initDefaultSpectrum(scene,'hyperspectral');
% 
% return;

%--------------------------------------------------
function scene = sceneMackay(scene,radFreq,sz)
% Someone (I think Chris Tyler) told me the ring/ray pattern is also called
% the Mackay chart. Reference from Joyce here:
%
% Some people call it the Siemens Star pattern (Wueller).
%
% We fill the central circle with a masking pattern.  The size of the
% central region is at the point when the rays would start to alias.  The
% the circumference of the central circle is 2*pi*r (with r in units of
% pixels).  When the radial frequency is f, we need a minimum of 2f pixels
% on the circumference.  So the circumference is 2*pi*r, so that we want
% the radius to be at least r = f/pi.  In practice that is too exact for
% the digital domain.  So we double the radius.
%

if ieNotDefined('radFreq'), radFreq = 8; end
if ieNotDefined('sz'),      sz = 256; end

scene = sceneSet(scene,'name','mackay');

if ~isfield(scene,'spectrum')
    scene = initDefaultSpectrum(scene,'hyperspectral');
end
nWave = sceneGet(scene,'nwave');

img = imgMackay(radFreq,sz);

% Insert central circle mask
r = round(2*radFreq/pi);  % Find the radius for the central circle

% Find the distance from the center of the image
[X,Y] = meshgrid(1:sz,1:sz); X = X - mean(X(:)); Y = Y - mean(Y(:));
d = sqrt(X.^2 + Y.^2);

% Everything with a distance less than 2r set to mean gray (128) for now.
l = (d < r);
img(l) = 128;  % figure; imagesc(img)

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));

return;


%--------------------------------------------------
function scene = sceneCzp(scene,radFreq,sz)

% We fill the central circle with a masking pattern.  The size of the
% central region is at the point when the rays would start to alias.  The
% the circumference of the central circle is 2*pi*r (with r in units of
% pixels).  When the radial frequency is f, we need a minimum of 2f pixels
% on the circumference.  So the circumference is 2*pi*r, so that we want
% the radius to be at least r = f/pi.  In practice that is too exact for
% the digital domain.  So we double the radius.
%

disp('In sceneCzp. . . .');

%{
if ieNotDefined('radFreq'), 
   disp('Radial frequency not defined. Setting to 8');
   radFreq = 8; 
end

if ieNotDefined('sz'),      
   disp('Image size not defined. Setting to 256');
   sz = 256; 
end
%}

scene = sceneSet(scene,'name','czp');

if ~isfield(scene,'spectrum')
    scene = initDefaultSpectrum(scene,'hyperspectral');
end
nWave = sceneGet(scene,'nwave');

img = imgCzp(radFreq,sz);

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));

return;

%--------------------------------------------------
function scene = sceneSweep(scene,sz,maxFreq)
%  These are always equal photon

if ieNotDefined('sz'), sz = 128; end
if ieNotDefined('maxFreq'), maxFreq = sz/16; end

scene = sceneSet(scene,'name','sweep');
scene = initDefaultSpectrum(scene,'hyperspectral');
nWave = sceneGet(scene,'nwave');

img = imgSweep(sz,maxFreq);

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));

return;

%--------------------------------------------------
function [scene,p] = sceneHarmonic(scene,parms)
%
% Possible parameters are:
% parms.freq, parms.row, parms.col, parms.ang
% parms.ph, parms.contrast
% If any are missing, defaults are supplied by imageHarmonic.

scene = sceneSet(scene,'name','harmonic');

scene = initDefaultSpectrum(scene,'hyperspectral');
nWave = sceneGet(scene,'nwave');

% TODO: Adjust pass the parameters back from the imgHarmonic window. In
% other cases, they are simply attached to the global parameters in
% vcSESSION.  We can get them by a getappdata call in here, but not if we
% close the window as part of imageSetHarmonic
if ieNotDefined('parms')
    global parms;
    h= imageSetHarmonic; waitfor(h);
    img = imageHarmonic(parms);
    p = parms;
    clear parms;
else
    [img,p] = imageHarmonic(parms);
end

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));

return;

%--------------------------------------------------
function scene = sceneRamp(scene,sz)
%

if ieNotDefined('sz'), sz = 128; end

scene = sceneSet(scene,'name','ramp');
scene = initDefaultSpectrum(scene,'hyperspectral');
nWave = sceneGet(scene,'nwave');

img = imgRamp(sz);

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));

return;

%--------------------------------------------------
function scene = sceneUniform(scene,spectralType,sz)
% Create a spatially uniform scene.
%

if ieNotDefined('scene'), error('Scene required.'); end
if ieNotDefined('spectralType'), spectralType = 'ee'; end
if ieNotDefined('sz'), sz = 32; end
scene = sceneSet(scene,'name',sprintf('uniform-%s',spectralType));

if ~isfield(scene,'spectrum')
    scene = initDefaultSpectrum(scene,'hyperspectral');
end

wave  = sceneGet(scene,'wave');
nWave = sceneGet(scene,'nwave');

d = ones(sz,sz,nWave);

switch lower(spectralType)
    case {'d65'}
        spect = vcReadSpectra('D65',wave);
    case {'ee','equalenergy','eenergy'}
        spect = Energy2Quanta(wave,ones(nWave,1));
    case {'ep','equalphoton','ephoton'}
        spect = ones(nWave,1);
    otherwise
        error('Unknown spectral type:%s\n',spectralType);
end

for ii=1:nWave, d(:,:,ii) = d(:,:,ii)*spect(ii); end
scene = sceneSet(scene,'cphotons',d);

return;

%--------------------------------------------------
function scene = sceneLine(scene,spectralType,sz,offset)
%
% Create a single line scene.  This is used for computing linespreads and
% OTFs.

if ieNotDefined('spectralType'), spectralType = 'ee'; end
if ieNotDefined('sz'),     sz = 64; end
if ieNotDefined('offset'), offset = 0; end

scene = sceneSet(scene,'name',sprintf('line-%s',spectralType));

if ~isfield(scene,'spectrum')
    scene = initDefaultSpectrum(scene,'hyperspectral');
end
wave    = sceneGet(scene,'wave');
nWave   = sceneGet(scene,'nwave');
photons = zeros(sz,sz,nWave);
linePos = round(sz/2) + offset;

% Figure out a way to do this using sceneSet.
switch lower(spectralType)
    case {'ep','equalphoton','ephoton'}
        % Equal number of photons at every wavelength
        photons(:,linePos,:) = 10^16;
    case {'ee','equalenergy','eenergy'}
        % Equal energy at every wavelength.  The large scale factor applied
        % to the number of photons is just to produce a reasonable energy
        % level. 
        p = Energy2Quanta(wave,10^16*ones(nWave,1));
        % Check that the photons have equal energy
        % e = Quanta2Energy(wave,p')
        for ii=1:nWave, photons(:,linePos,ii) = p(ii); end
    case 'd65'
        % D65 spectra for the line
        d65 = vcReadSpectra('D65',wave);
        for ii=1:nWave, photons(:,linePos,ii) = d65(ii); end
    otherwise
        error('Unknown uniform field type.');
end

scene = sceneSet(scene,'cphotons',photons);

return;

%------------------------
function scene = sceneRadialLines(scene,imSize,spectralType,nLines)
% Create a Siemens Star (radial line) scene.
%
%   scene = sceneRadialLines(scene,imSize,spectralType,nLines)
%
% In this test chart the intensities along lines from the center are
% constant. Measuring on a circle around the center the intensity is a
% harmonic. Hence, frequency varies as a function of radial distance.
%
% Reference:
%   Dieter Wueller thinks this pattern is cool.
%   Digital camera resolution measurement using sinusoidal Siemens stars
%   Proc. SPIE, Vol. 6502, 65020N (2007); doi:10.1117/12.703817
%
% Examples:
%  scene = sceneCreate('radialLines');
%

if ieNotDefined('scene'), error('Scene must be defined'); end
if ieNotDefined('spectralType'), spectralType = 'ee'; end
if ieNotDefined('imSize'), imSize = 256; end
if ieNotDefined('nLines'), nLines = 8; end

scene = sceneSet(scene,'name',sprintf('radialLine-%s',spectralType));
scene = initDefaultSpectrum(scene,'hyperspectral');

% Determine the line angles
radians = pi*(0:(nLines-1))/nLines;
endPoints = zeros(nLines,2);
for ii=1:nLines
    endPoints(ii,:) = round([cos(radians(ii)),sin(radians(ii))]*imSize/2);
end
% plot(endPoints(:,1),endPoints(:,2),'o')

img = zeros(imSize,imSize);

% The routine for drawing lines could be better.
for ii=1:nLines
    x = endPoints(ii,1); y = endPoints(ii,2);
    u = -x; v = -y;
    % Flip so x is the lower one
    if x > 0,
        tmp = [x,y]; x = u; y = v; u = tmp(1); v = tmp(2);
    end

    if ~isequal(u,x), slope = (y - v) / (u - x);
        for jj=x:0.2:u,
            kk = round(jj*slope);
            img(round(kk + (imSize/2)) + 1, round(jj + (imSize/2)) + 1) = 1;
        end
    else img(:, (imSize/2) + 1) = 1;
    end
end
img = img(1:imSize,1:imSize);
% figure; imagesc(img)

% Create the photon image
wave = sceneGet(scene,'wave');
nWave = sceneGet(scene,'nwave');
photons = zeros(imSize,imSize,nWave);

% Figure out a way to do this using sceneSet.
switch lower(spectralType)
    case {'ep','equalphoton','ephoton'}
        % Equal number of photons at every wavelength
        img = img*10^16;
        for ii=1:nWave, photons(:,:,ii) = img; end
    case {'ee','equalenergy','eenergy'}
        % Equal energy at every wavelength.  The large scale factor applied
        % to the number of photons is just to produce a reasonable energy
        % level.
        p = Energy2Quanta(wave,10^16*ones(nWave,1));
        % Check that the photons have equal energy
        % e = Quanta2Energy(wave,p')
        for ii=1:nWave, photons(:,:,ii) = p(ii)*img; end
    case 'd65'
        % D65 spectra for the line
        d65 = vcReadSpectra('D65',wave);
        for ii=1:nWave, photons(:,:,ii) = d65(ii)*img; end
    otherwise
        error('Unknown uniform field type.');
end

scene = sceneSet(scene,'cphotons',photons);

return;

%-----------------------
function scene = sceneFOTarget(scene,parms)
%
% Frequency/Orientation target

if ieNotDefined('parms'), parms = []; end

scene = sceneSet(scene,'name','FOTarget');
scene = initDefaultSpectrum(scene,'hyperspectral');
nWave = sceneGet(scene,'nwave');

img = FOTarget('sine',parms);

% This routine returns an RGB image.  We take the green channel and expand
% it
scene = sceneSet(scene,'cphotons',repmat(img(:,:,2),[1,1,nWave]));

return;

%-------------------
function scene = sceneCheckerboard(scene,checkPeriod,nCheckPairs,spectralType)
%
% Checkerboard

if ieNotDefined('scene'), error('Scene required'); end
if ieNotDefined('checkPeriod'), checkPeriod = 16; end
if ieNotDefined('nCheckPairs'), nCheckPairs = 8; end
if ieNotDefined('spectralType'), spectralType = 'ep'; end

scene = sceneSet(scene,'name',sprintf('Checker-%s',spectralType));
scene = initDefaultSpectrum(scene,'hyperspectral');
wave  = sceneGet(scene,'wave');
nWave = sceneGet(scene,'nwave');

d = checkerboard(checkPeriod,nCheckPairs); 

switch lower(spectralType)
    case {'d65'}
        spect = vcReadSpectra('D65',wave);
    case {'ee','equalenergy'}
        spect = Energy2Quanta(wave,ones(nWave,1));
    case {'ep','equalphoton'}
        spect = ones(nWave,1);
    otherwise
        error('Unknown spectral type:%s\n',spectralType);
end

img = zeros(size(d,1),size(d,2),nWave);
for ii=1:nWave, img(:,:,ii) = d*spect(ii); end

% This routine returns an RGB image.  We take the green channel and expand
% it
scene = sceneSet(scene,'cphotons',img);

return;

%---------------------------------------------------------------
function scene = sceneSlantedBar(scene,imSize,barSlope,fieldOfView)
%
%  Slanted bar, 2 deg field of view
%

if ieNotDefined('imSize'),      imSize = 384; end
if ieNotDefined('barSlope'),    barSlope = 2.6; end
if ieNotDefined('fieldOfView'), fieldOfView = 2; end

scene = sceneSet(scene,'name','slantedBar');

if ~isfield(scene,'spectrum')
    scene = initDefaultSpectrum(scene,'hyperspectral');
end
nWave = sceneGet(scene,'nwave');

imSize = round(imSize/2);
[X,Y] = meshgrid(-imSize:imSize,-imSize:imSize);
img = zeros(size(X));

%  y = barSlope*x defines the line.  We find all the Y values that are
%  above the line
list = (Y > barSlope*X );
img( list ) = 1;

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));
scene = sceneSet(scene,'horizontalfieldofview',fieldOfView);

return;

%-----------------------
function scene = sceneZonePlate(scene,imSize,fieldOfView)
%
% Circular zone plate image
%

if ieNotDefined('imSize'), imSize = 256; end
if ieNotDefined('fieldOfView'), fieldOfView = 4; end

scene = sceneSet(scene,'name','zonePlate');
scene = initDefaultSpectrum(scene,'hyperspectral');
nWave = sceneGet(scene,'nwave');

img = imgZonePlate(imSize); 

scene = sceneSet(scene,'cphotons',repmat(img,[1,1,nWave]));
scene = sceneSet(scene,'horizontalfieldofview',fieldOfView);

return;

